#!/usr/bin/env python3
# -*- coding:utf-8 -*-
"""
SQL Database Reporter for YOLOX Traffic Analysis
Handles connection to SQL Server and updates ReportTable with traffic data
"""

import pymysql
from loguru import logger
from typing import Dict, Any, Optional
import sys


class SQLReporter:
    """Handles MySQL RDS connections and traffic data reporting to ReportTable2"""
    
    def __init__(self):
        # DB Connection Information (Updated for MySQL RDS)
        self.sql_server = "reportoutput.cqp6e2m0m3zl.us-east-1.rds.amazonaws.com"
        self.sql_database = "mobius"
        self.sql_username = "admin"
        self.sql_password = "0*JZON5z?4)XWz7[[c_f-b59Hn0p"
        self.connection = None
        
    def connect(self) -> bool:
        """Establish connection to MySQL RDS"""
        try:
            self.connection = pymysql.connect(
                host=self.sql_server,
                port=3306,
                user=self.sql_username,
                password=self.sql_password,
                database=self.sql_database,
                cursorclass=pymysql.cursors.DictCursor,
                connect_timeout=30
            )
            logger.info("Successfully connected to MySQL RDS")
            return True
            
        except Exception as e:
            logger.error(f"Failed to connect to MySQL RDS: {e}")
            return False
    
    def disconnect(self):
        """Close MySQL connection"""
        if self.connection:
            self.connection.close()
            self.connection = None
            logger.info("Disconnected from MySQL")
    
    def OLD_insert_traffic_data(self, traffic_data: Dict[str, Any]) -> bool:
        """
        Insert traffic analysis data into dbo.ReportTable
        
        Args:
            traffic_data: Dictionary containing all traffic counts and metadata
        """
        if not self.connection:
            logger.error("No database connection available")
            return False
        
        try:
            cursor = self.connection.cursor()
            
            # Build the INSERT query
            insert_query = """
                INSERT INTO `ReportTable` (
                    `File_Name`,
                    `File_Instance`,
                    `Time`,
                    `inTop_car`,
                    `inTop_Motorcycle`,
                    `inTop_Bus`,
                    `inTop_Truck`,
                    `inLeft_car`,
                    `inLeft_Motorcycle`,
                    `inLeft_Bus`,
                    `inLeft_Truck`,
                    `inRight_car`,
                    `inRight_Motorcycle`,
                    `inRight_Bus`,
                    `inRight_Truck`,
                    `inBottom_car`,
                    `inBottom_Motorcycle`,
                    `inBottom_Bus`,
                    `inBottom_Truck`,
                    `outTop_car`,
                    `outTop_Motorcycle`,
                    `outTop_Bus`,
                    `outTop_Truck`,
                    `outLeft_car`,
                    `outLeft_Motorcycle`,
                    `outLeft_Bus`,
                    `outLeft_Truck`,
                    `outRight_car`,
                    `outRight_Motorcycle`,
                    `outRight_Bus`,
                    `outRight_Truck`,
                    `outBottom_car`,
                    `outBottom_Motorcycle`,
                    `outBottom_Bus`,
                    `outBottom_Truck`
                ) VALUES (%s, %s, %s, %s, %s, %s, %s, %s, %s, %s, %s, %s, %s, %s, %s, %s, %s, %s, %s, %s, %s, %s, %s, %s, %s, %s, %s, %s, %s, %s, %s, %s, %s, %s, %s)
            """
            
            # Extract values in the correct order
            values = (
                traffic_data.get('file_name', ''),
                traffic_data.get('file_instance', 0),
                traffic_data.get('time_calc', ''),
                # Incoming counts
                traffic_data.get('inTop_car', 0),
                traffic_data.get('inTop_motorcycle', 0),
                traffic_data.get('inTop_bus', 0),
                traffic_data.get('inTop_truck', 0),
                traffic_data.get('inLeft_car', 0),
                traffic_data.get('inLeft_motorcycle', 0),
                traffic_data.get('inLeft_bus', 0),
                traffic_data.get('inLeft_truck', 0),
                traffic_data.get('inRight_car', 0),
                traffic_data.get('inRight_motorcycle', 0),
                traffic_data.get('inRight_bus', 0),
                traffic_data.get('inRight_truck', 0),
                traffic_data.get('inBottom_car', 0),
                traffic_data.get('inBottom_motorcycle', 0),
                traffic_data.get('inBottom_bus', 0),
                traffic_data.get('inBottom_truck', 0),
                # Outgoing counts
                traffic_data.get('outTop_car', 0),
                traffic_data.get('outTop_motorcycle', 0),
                traffic_data.get('outTop_bus', 0),
                traffic_data.get('outTop_truck', 0),
                traffic_data.get('outLeft_car', 0),
                traffic_data.get('outLeft_motorcycle', 0),
                traffic_data.get('outLeft_bus', 0),
                traffic_data.get('outLeft_truck', 0),
                traffic_data.get('outRight_car', 0),
                traffic_data.get('outRight_motorcycle', 0),
                traffic_data.get('outRight_bus', 0),
                traffic_data.get('outRight_truck', 0),
                traffic_data.get('outBottom_car', 0),
                traffic_data.get('outBottom_motorcycle', 0),
                traffic_data.get('outBottom_bus', 0),
                traffic_data.get('outBottom_truck', 0)
            )
            
            cursor.execute(insert_query, values)
            self.connection.commit()
            
            logger.info(f"Successfully inserted traffic data for {traffic_data.get('file_name', 'unknown')}")
            return True
            
        except Exception as e:
            logger.error(f"Failed to insert traffic data: {e}")
            if self.connection:
                self.connection.rollback()
            return False
    
    def OLD_update_traffic_data(self, traffic_data: Dict[str, Any], where_clause: str) -> bool:
        """
        Update existing traffic analysis data in dbo.ReportTable
        
        Args:
            traffic_data: Dictionary containing traffic counts to update
            where_clause: SQL WHERE clause to identify the row(s) to update
        """
        if not self.connection:
            logger.error("No database connection available")
            return False
        
        try:
            cursor = self.connection.cursor()
            
            # Build the UPDATE query dynamically based on provided data
            update_fields = []
            values = []
            
            field_mapping = {
                'file_name': '`File_Name`',
                'file_instance': '`File_Instance`',
                'time_calc': '`Time`',
                # Incoming counts
                'inTop_car': '`inTop_car`',
                'inTop_motorcycle': '`inTop_Motorcycle`',
                'inTop_bus': '`inTop_Bus`',
                'inTop_truck': '`inTop_Truck`',
                'inLeft_car': '`inLeft_car`',
                'inLeft_motorcycle': '`inLeft_Motorcycle`',
                'inLeft_bus': '`inLeft_Bus`',
                'inLeft_truck': '`inLeft_Truck`',
                'inRight_car': '`inRight_car`',
                'inRight_motorcycle': '`inRight_Motorcycle`',
                'inRight_bus': '`inRight_Bus`',
                'inRight_truck': '`inRight_Truck`',
                'inBottom_car': '`inBottom_car`',
                'inBottom_motorcycle': '`inBottom_Motorcycle`',
                'inBottom_bus': '`inBottom_Bus`',
                'inBottom_truck': '`inBottom_Truck`',
                # Outgoing counts
                'outTop_car': '`outTop_car`',
                'outTop_motorcycle': '`outTop_Motorcycle`',
                'outTop_bus': '`outTop_Bus`',
                'outTop_truck': '`outTop_Truck`',
                'outLeft_car': '`outLeft_car`',
                'outLeft_motorcycle': '`outLeft_Motorcycle`',
                'outLeft_bus': '`outLeft_Bus`',
                'outLeft_truck': '`outLeft_Truck`',
                'outRight_car': '`outRight_car`',
                'outRight_motorcycle': '`outRight_Motorcycle`',
                'outRight_bus': '`outRight_Bus`',
                'outRight_truck': '`outRight_Truck`',
                'outBottom_car': '`outBottom_car`',
                'outBottom_motorcycle': '`outBottom_Motorcycle`',
                'outBottom_bus': '`outBottom_Bus`',
                'outBottom_truck': '`outBottom_Truck`'
            }
            
            for key, value in traffic_data.items():
                if key in field_mapping:
                    update_fields.append(f"{field_mapping[key]} = %s")
                    values.append(value)
            
            if not update_fields:
                logger.warning("No valid fields to update")
                return False
            
            update_query = f"""
                UPDATE `ReportTable`
                SET {', '.join(update_fields)}
                WHERE {where_clause}
            """
            
            cursor.execute(update_query, values)
            rows_affected = cursor.rowcount
            self.connection.commit()
            
            logger.info(f"Successfully updated {rows_affected} row(s) in ReportTable")
            return True
            
        except Exception as e:
            logger.error(f"Failed to update traffic data: {e}")
            if self.connection:
                self.connection.rollback()
            return False
    
    def OLD_get_existing_record(self, file_name: str, file_instance: int) -> Optional[Dict]:
        """
        Check if a record already exists for the given file_name and file_instance
        
        Args:
            file_name: Name of the processed file
            file_instance: Instance number of the file
            
        Returns:
            Dictionary with existing record data or None if not found
        """
        if not self.connection:
            logger.error("No database connection available")
            return None
        
        try:
            cursor = self.connection.cursor()
            
            query = """
                SELECT * FROM `ReportTable`
                WHERE `File_Name` = %s AND `File_Instance` = %s
            """
            
            cursor.execute(query, (file_name, file_instance))
            row = cursor.fetchone()
            
            if row:
                # Convert row to dictionary
                columns = [column[0] for column in cursor.description]
                return dict(zip(columns, row))
            
            return None
            
        except Exception as e:
            logger.error(f"Failed to check existing record: {e}")
            return None
    
    def get_existing_record_table2(self, file_name: str, file_instance: int) -> Optional[Dict]:
        """
        Check if a record already exists in ReportTable2
        
        Args:
            file_name: Name of the processed file
            file_instance: Instance number of the file
            
        Returns:
            Dictionary with existing record data or None if not found
        """
        if not self.connection:
            logger.error("No database connection available")
            return None
        
        try:
            cursor = self.connection.cursor()
            
            query = """
                SELECT * FROM `ReportTable2`
                WHERE `File_Name` = %s AND `File_Instance` = %s
            """
            
            cursor.execute(query, (file_name, file_instance))
            row = cursor.fetchone()
            
            if row:
                # Convert row to dictionary (already dict with DictCursor)
                return row
            
            return None
            
        except Exception as e:
            logger.error(f"Failed to check existing record in ReportTable2: {e}")
            return None
    
    def update_turning_movement_data(self, traffic_data: Dict[str, Any], file_name: str, file_instance: int) -> bool:
        """
        Update existing turning movement data in ReportTable2
        
        Args:
            traffic_data: Dictionary containing turning movement data
            file_name: File name to update
            file_instance: Instance number to update
        """
        if not self.connection:
            logger.error("No database connection available")
            return False
        
        try:
            cursor = self.connection.cursor()
            
            # Extract turning movements
            turning_movements = traffic_data.get('turning_movements', {})
            
            # Build UPDATE query
            update_parts = []
            values = []
            
            # Update Time field
            update_parts.append("`Time` = %s")
            values.append(traffic_data.get('time_calc', ''))
            
            # Update turning movement columns
            for movement_key, count in turning_movements.items():
                update_parts.append(f"`{movement_key}` = %s")
                values.append(count)
            
            # Update summary columns
            update_parts.extend([
                "`Total_Cars` = %s",
                "`Total_Motorcycles` = %s",
                "`Total_Buses` = %s",
                "`Total_Trucks` = %s",
                "`Total_Vehicles` = %s"
            ])
            values.extend([
                traffic_data.get('Total_Cars', 0),
                traffic_data.get('Total_Motorcycles', 0),
                traffic_data.get('Total_Buses', 0),
                traffic_data.get('Total_Trucks', 0),
                traffic_data.get('Total_Vehicles', 0)
            ])
            
            # Add WHERE clause parameters
            values.extend([file_name, file_instance])
            
            update_query = f"""
                UPDATE `ReportTable2`
                SET {', '.join(update_parts)}
                WHERE `File_Name` = %s AND `File_Instance` = %s
            """
            
            cursor.execute(update_query, values)
            rows_affected = cursor.rowcount
            self.connection.commit()
            
            logger.info(f"Successfully updated {rows_affected} row(s) in ReportTable2")
            return True
            
        except Exception as e:
            logger.error(f"Failed to update turning movement data: {e}")
            if self.connection:
                self.connection.rollback()
            return False
    
    def insert_turning_movement_data(self, traffic_data: Dict[str, Any]) -> bool:
        """
        Insert traffic data into ReportTable2 with turning movement columns
        
        Args:
            traffic_data: Dictionary containing turning movement data
        """
        if not self.connection:
            logger.error("No database connection available")
            return False
        
        try:
            cursor = self.connection.cursor()
            
            # Extract turning movements
            turning_movements = traffic_data.get('turning_movements', {})
            
            # Build column list and values
            columns = ['`File_Name`', '`File_Instance`', '`Time`']
            values = [
                traffic_data.get('file_name', ''),
                traffic_data.get('file_instance', 0),
                traffic_data.get('time_calc', '')
            ]
            
            # Add turning movement columns
            for movement_key, count in turning_movements.items():
                # Convert movement key to column name
                # e.g., "North_to_East_car" stays as is
                column_name = f"`{movement_key}`"
                columns.append(column_name)
                values.append(count)
            
            # Add summary columns
            columns.extend(['`Total_Cars`', '`Total_Motorcycles`', '`Total_Buses`', '`Total_Trucks`', '`Total_Vehicles`'])
            values.extend([
                traffic_data.get('Total_Cars', 0),
                traffic_data.get('Total_Motorcycles', 0),
                traffic_data.get('Total_Buses', 0),
                traffic_data.get('Total_Trucks', 0),
                traffic_data.get('Total_Vehicles', 0)
            ])
            
            # Build INSERT query (PyMySQL uses %s for placeholders)
            placeholders = ', '.join(['%s'] * len(values))
            columns_str = ', '.join(columns)
            
            insert_query = f"""
                INSERT INTO `ReportTable2` ({columns_str})
                VALUES ({placeholders})
            """
            
            cursor.execute(insert_query, values)
            self.connection.commit()
            
            logger.info(f"Successfully inserted turning movement data for {traffic_data.get('file_name', 'unknown')}")
            return True
            
        except Exception as e:
            logger.error(f"Failed to insert turning movement data: {e}")
            if self.connection:
                self.connection.rollback()
            return False
    
    def report_traffic_analysis(self, traffic_data: Dict[str, Any]) -> bool:
        """
        Main method to report traffic analysis results to ReportTable2
        Only uses ReportTable2 - no fallback to old table
        Will update existing record or insert new one
        
        Args:
            traffic_data: Complete traffic analysis data dictionary with turning movements
        """
        if not self.connect():
            return False
        
        try:
            file_name = traffic_data.get('file_name', '')
            file_instance = traffic_data.get('file_instance', 0)
            
            # Check if record already exists in ReportTable2
            existing_record = self.get_existing_record_table2(file_name, file_instance)
            
            if existing_record:
                # Update existing record
                logger.info(f"Updating existing record in ReportTable2 for {file_name}, instance {file_instance}")
                success = self.update_turning_movement_data(traffic_data, file_name, file_instance)
            else:
                # Insert new record
                logger.info(f"Inserting new record to ReportTable2 for {file_name}, instance {file_instance}")
                success = self.insert_turning_movement_data(traffic_data)
            
            return success
            
        finally:
            self.disconnect()


def test_connection():
    """Test the SQL connection"""
    reporter = SQLReporter()
    if reporter.connect():
        logger.info("SQL connection test successful")
        reporter.disconnect()
        return True
    else:
        logger.error("SQL connection test failed")
        return False


if __name__ == "__main__":
    # Test connection when run directly
    test_connection()